<?php
/**
 * ===================================================================================
 * My Private Files Page (for Students)
 * ===================================================================================
 *
 * File: my_private_files.php
 * Location: /homework_portal/student/my_private_files.php
 *
 * Purpose:
 * 1. Displays a list of private files uploaded by teachers for the logged-in student.
 * 2. Automatically marks these files as 'read' upon viewing the page.
 */

require_once '../config.php';

// --- Security Check & Initial Setup ---
if (!isset($_SESSION['student_id'])) {
    header('Location: ../login.php');
    exit();
}

$student_id = $_SESSION['student_id'];

try {
    // 1. Fetch all private files for this student, joining with the teacher's name
    $files_sql = "
        SELECT 
            pf.id, pf.title, pf.notes, pf.file_path, pf.created_at,
            t.full_name as teacher_name
        FROM student_private_files pf
        JOIN teachers t ON pf.teacher_id = t.id
        WHERE pf.student_id = ? 
        ORDER BY pf.created_at DESC
    ";
    $files_stmt = $pdo->prepare($files_sql);
    $files_stmt->execute([$student_id]);
    $private_files = $files_stmt->fetchAll();

    // 2. After fetching, mark all of this student's unread files as 'read'
    // This clears the notification badge for the teacher.
    if (!empty($private_files)) {
        $update_sql = "UPDATE student_private_files SET is_read = 1 WHERE student_id = ? AND is_read = 0";
        $pdo->prepare($update_sql)->execute([$student_id]);
    }

} catch (PDOException $e) {
    $private_files = [];
    $page_error = "We had a problem loading your private files. Please tell your teacher!";
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Private Files - School Portal</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@400;600;700&family=Baloo+2:wght@700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Poppins', sans-serif; background-color: #f0f4f8; }
        .font-brand { font-family: 'Baloo 2', cursive; }
        .file-card {
            background-color: white;
            border-left: 5px solid #4dd0e1; /* Bright cyan accent */
            border-radius: 8px;
            padding: 1.5rem;
            box-shadow: 0 4px 15px rgba(0,0,0,0.05);
        }
    </style>
</head>
<body class="min-h-screen">

    <!-- Header -->
    <header class="bg-white shadow-sm sticky top-0 z-10">
        <div class="max-w-5xl mx-auto px-4 sm:px-6 lg:px-8 py-4 flex justify-between items-center">
            <a href="dashboard.php" class="flex items-center text-cyan-700 hover:text-cyan-800 font-medium">
                 <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"></path></svg>
                Back to Dashboard
            </a>
            <a href="../logout.php" class="py-2 px-4 bg-orange-500 text-white text-sm font-semibold rounded-full shadow-md hover:bg-orange-600">Logout</a>
        </div>
    </header>

    <!-- Main Content -->
    <main class="max-w-3xl mx-auto py-8 px-4 sm:px-6 lg:px-8">
        <div class="mb-8 text-center">
            <h1 class="text-4xl font-bold font-brand text-gray-800">My Private Files</h1>
            <p class="text-gray-500">Important documents from your teachers.</p>
        </div>

        <?php if (isset($page_error)): ?>
            <div class="p-4 text-center text-red-800 bg-red-100 rounded-lg"><?php echo e($page_error); ?></div>
        <?php elseif (empty($private_files)): ?>
             <div class="p-6 text-center text-gray-600 bg-white rounded-lg shadow">
                <p>You have no private files from your teachers right now.</p>
            </div>
        <?php else: ?>
            <!-- Files List -->
            <div class="space-y-4">
                <?php foreach ($private_files as $file): ?>
                    <div class="file-card">
                        <div class="flex justify-between items-start">
                            <h3 class="font-bold text-lg text-gray-800"><?php echo e($file['title']); ?></h3>
                            <span class="text-xs text-gray-400"><?php echo date('M d, Y', strtotime($file['created_at'])); ?></span>
                        </div>
                         <p class="text-sm text-gray-500 mb-3">Uploaded by: <?php echo e($file['teacher_name']); ?></p>
                         <?php if(!empty($file['notes'])): ?>
                            <p class="mt-2 text-gray-600 bg-gray-50 p-3 rounded-md"><?php echo nl2br(e($file['notes'])); ?></p>
                         <?php endif; ?>
                         <div class="mt-4">
                            <a href="../<?php echo e($file['file_path']); ?>" download class="inline-flex items-center px-4 py-2 bg-blue-500 text-white text-sm font-semibold rounded-lg hover:bg-blue-600">
                                <svg class="w-5 h-5 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4"></path></svg>
                                Download File
                            </a>
                         </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </main>

</body>
</html>
